/**  @file bta_crop_chessboard.c
*
*    @brief This file implements the filter (see header)
*
*    BLT_DISCLAIMER
*
*    @author Alex Falkensteiner
*
*    @cond svn
*
*    Information of last commit
*    $Rev::               $:  Revision of last commit
*    $Author::            $:  Author of last commit
*    $Date::              $:  Date of last commit
*
*    @endcond
*/


#include "bta_crop_chessboard.h"
#include <bta_opencv_helper.h>
#include <bta_helper.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#ifndef BTA_EXCLUDE_FILTERS

#include <opencv2/core/core.hpp>
#include <opencv2/imgproc/imgproc.hpp>
#include <opencv2/calib3d/calib3d.hpp>


static std::vector<cv::Point2f> scaleVector(std::vector<cv::Point2f> pointBuf, float resize);


BTA_Status BFLTcropChessboardInit(BTA_FltCropChessboardConfig *config, BTA_FltHandle *handle, BTA_InfoEventInst *infoEventInst) {
    if (!handle || !config) {
        return BTA_StatusInvalidParameter;
    }
    *handle = 0;
    BTA_FltCropChessboardInst *inst = (BTA_FltCropChessboardInst *)calloc(1, sizeof(BTA_FltCropChessboardInst));
    if (!inst) {
        return BTA_StatusOutOfMemory;
    }
    inst->channelToProcess = config->channelToProcess;
    inst->scaleFactor = config->scaleFactor;
    inst->edgeCountHor = config->edgeCountHor;
    inst->edgeCountVert = config->edgeCountVert;
    inst->border = config->border;
    inst->channelIdResult= config->channelIdResult;
    inst->infoEventInst = infoEventInst;
    *handle = inst;
    return BTA_StatusOk;
}


BTA_Status BFLTcropChessboardClose(BTA_FltHandle *handle) {
    BTA_FltCropChessboardInst **inst = (BTA_FltCropChessboardInst **)handle;
    free(*inst);
    *inst = 0;
    return BTA_StatusOk;
}


BTA_Status BFLTcropChessboardApply(BTA_FltHandle handle, BTA_Frame **frame) {
    BTA_FltCropChessboardInst *inst = (BTA_FltCropChessboardInst *)handle;
    int chInd;
    int chCount = (*frame)->channelsLen;
    for (chInd = 0; chInd < chCount; chInd++) {
        BTA_Channel *channel = ((*frame)->channels)[chInd];
        if (channel->id == inst->channelToProcess) {
            uint8_t freeChannel = 0;
            if (channel->dataFormat == BTA_DataFormatRgb24 || channel->dataFormat == BTA_DataFormatRgb565) {
                BTA_Channel *channelTemp;
                BTA_Status status = BTAcolorToBw(channel, &channelTemp);
                if (status != BTA_StatusOk) {
                    return status;
                }
                channel = channelTemp;
                freeChannel = 1;
            }
            cv::Mat channelMat = BTAtoMat(channel);
            double minOrig, maxOrig;
            cv::minMaxLoc(channelMat, &minOrig, &maxOrig);
            cv::Mat channelMat8;
            if (std::abs(minOrig) >= 1 && (maxOrig - minOrig != 0)) {
                channelMat.convertTo(channelMat8, CV_8U, 255.0 / (maxOrig - minOrig), -255.0 / minOrig);
            }
            else {
                channelMat.convertTo(channelMat8, CV_8U, 255.0 / maxOrig);
            }
            cv::Mat channelMatUpscaled;
            if (inst->scaleFactor != 1) {
                cv::resize(channelMat8, channelMatUpscaled, cv::Size(0, 0), inst->scaleFactor, inst->scaleFactor);
            }
            else {
                channelMatUpscaled = channelMat8;
            }
            cv::Size s;
            s.width = inst->edgeCountHor;
            s.height = inst->edgeCountVert;
            std::vector<cv::Point2f> pointBuf;
            bool found = findChessboardCorners(channelMatUpscaled, s, pointBuf, CV_CALIB_CB_ADAPTIVE_THRESH | CV_CALIB_CB_FAST_CHECK);
            if (freeChannel) {
                // we created a new intermediate channel before -> free it!
                BTAfreeChannel(&channel);
                channel = ((*frame)->channels)[chInd];
            }
            if (!found) {
                if (!inst->channelIdResult) {
                    uint8_t * dataDummy = (uint8_t *)calloc(8, sizeof(uint8_t)); // at least 8 byte for biggest dataFormat
                    *dataDummy = 0;
                    free(channel->data);
                    channel->data = dataDummy;
                    channel->xRes = 1;
                    channel->yRes = 1;
                    channel->dataLen = channel->xRes * channel->yRes * (channel->dataFormat & 0xf);
                }
                //uint8_t * dataDummy = (uint8_t *)malloc(1);
                //*dataDummy = 0;
                //BTAinsertChannelDataIntoFrame((*frame), inst->channelIdResult, 1, 1, channelTemp->dataFormat, channelTemp->unit, 0, 0, dataDummy, channelTemp->dataFormat & 0xf);
                continue;
            }
            cornerSubPix(channelMatUpscaled, pointBuf, cv::Size(5, 5), cv::Size(-1, -1), cv::TermCriteria(CV_TERMCRIT_EPS + CV_TERMCRIT_ITER, 30, 0.1));

            std::vector<cv::Point2f> points = scaleVector(pointBuf, 1.0f / inst->scaleFactor);
            int i;
            //for (i = 0; i < pointBuf.size(); i++) {
            //    ((uint16_t *)channelTemp->data)[(int)points[i].x + (int)points[i].y * channelTemp->xRes] = 10000;
            //}
            float xfMax = 0, xfMin = channel->xRes;
            float yfMax = 0, yfMin = channel->yRes;
            for (i = 0; i < s.width * s.height; i++) {
                xfMax = (float)fmax(xfMax, points[i].x);
                xfMin = (float)fmin(xfMin, points[i].x);
                yfMax = (float)fmax(yfMax, points[i].y);
                yfMin = (float)fmin(yfMin, points[i].y);
            }
            int xMin = MAX(0, (int)roundf(xfMin) - (int)inst->border);
            int xMax = MIN(channel->xRes - 1, (int)roundf(xfMax) + (int)inst->border);
            if (s.width > 1) {
                xMin = MAX(0, (int)roundf(xfMin - (1 + inst->border)*(xfMax - xfMin) / (s.width - 1)));
                xMax = MIN(channel->xRes - 1, (int)roundf(xfMax + (1 + inst->border)*(xfMax - xfMin) / (s.width - 1)));
            }
            while (xMin > xMax) {
                xMin = MAX(0, xMin - 1);
                xMax = MIN(channel->xRes - 1, xMax + 1);
            }
            int yMin = MAX(0, (int)roundf(yfMin) - (int)inst->border);
            int yMax = MIN(channel->yRes - 1, (int)roundf(yfMax) + (int)inst->border);
            if (s.height > 1) {
                yMin = MAX(0, (int)roundf(yfMin - (1 + inst->border)*(yfMax - yfMin) / (s.height - 1)));
                yMax = MIN(channel->yRes - 1, (int)roundf(yfMax + (1 + inst->border)*(yfMax - yfMin) / (s.height - 1)));
            }
            while (yMin > yMax) {
                yMin = MAX(0, yMin - 1);
                yMax = MIN(channel->yRes - 1, yMax + 1);
            }
            int x, y;
            i = 0;
            switch (channel->dataFormat) {

            case BTA_DataFormatUInt16:
            case BTA_DataFormatSInt16:
            case BTA_DataFormatRgb565: {
                uint16_t *dataNew;
                if (inst->channelIdResult) {
                    dataNew = (uint16_t *)malloc((xMax - xMin + 1) * (yMax - yMin + 1) * sizeof(uint16_t));
                }
                else {
                    dataNew = (uint16_t *)channel->data;
                }
                for (y = yMin; y <= yMax; y++) {
                    for (x = xMin; x <= xMax; x++) {
                        dataNew[i++] = ((uint16_t *)channel->data)[x + y * channel->xRes];
                    }
                }
                if (inst->channelIdResult) {
                    BTAinsertChannelDataIntoFrame((*frame), inst->channelIdResult, xMax - xMin + 1, yMax - yMin + 1, channel->dataFormat, channel->unit, channel->integrationTime, channel->modulationFrequency, (uint8_t *)dataNew, (xMax - xMin + 1) * (yMax - yMin + 1) * sizeof(uint16_t));
                }
                else {
                    channel->xRes = xMax - xMin + 1;
                    channel->yRes = yMax - yMin + 1;
                    channel->dataLen = channel->xRes * channel->yRes * sizeof(uint16_t);
                }
                break;
            }

            case BTA_DataFormatFloat32: {
                float *dataNew;
                if (inst->channelIdResult) {
                    dataNew = (float *)malloc((xMax - xMin + 1) * (yMax - yMin + 1) * sizeof(float));
                }
                else {
                    dataNew = (float *)channel->data;
                }
                for (y = yMin; y <= yMax; y++) {
                    for (x = xMin; x <= xMax; x++) {
                        dataNew[i++] = ((float *)channel->data)[x + y * channel->xRes];
                    }
                }
                if (inst->channelIdResult) {
                    BTAinsertChannelDataIntoFrame((*frame), inst->channelIdResult, xMax - xMin + 1, yMax - yMin + 1, channel->dataFormat, channel->unit, channel->integrationTime, channel->modulationFrequency, (uint8_t *)dataNew, (xMax - xMin + 1) * (yMax - yMin + 1) * sizeof(float));
                }
                else {
                    channel->xRes = xMax - xMin + 1;
                    channel->yRes = yMax - yMin + 1;
                    channel->dataLen = channel->xRes * channel->yRes * sizeof(float);
                }
                break;
            }

            case BTA_DataFormatRgb24: {
                uint8_t *dataNew;
                if (inst->channelIdResult) {
                    dataNew = (uint8_t *)malloc((xMax - xMin + 1) * (yMax - yMin + 1) * 3);
                }
                else {
                    dataNew = (uint8_t *)channel->data;
                }
                for (y = yMin; y <= yMax; y++) {
                    for (x = xMin; x <= xMax; x++) {
                        dataNew[i++] = ((uint8_t *)channel->data)[3 * (x + y * channel->xRes) + 0];
                        dataNew[i++] = ((uint8_t *)channel->data)[3 * (x + y * channel->xRes) + 1];
                        dataNew[i++] = ((uint8_t *)channel->data)[3 * (x + y * channel->xRes) + 2];
                    }
                }
                if (inst->channelIdResult) {
                    BTAinsertChannelDataIntoFrame((*frame), inst->channelIdResult, xMax - xMin + 1, yMax - yMin + 1, channel->dataFormat, channel->unit, channel->integrationTime, channel->modulationFrequency, (uint8_t *)dataNew, (xMax - xMin + 1) * (yMax - yMin + 1) * 3);
                }
                else {
                    channel->xRes = xMax - xMin + 1;
                    channel->yRes = yMax - yMin + 1;
                    channel->dataLen = channel->xRes * channel->yRes * 3;
                }
                break;
            }

            default:
                BTAinfoEventHelper(inst->infoEventInst, 3, BTA_StatusNotSupported, "BFLTcropChessboardApply: unsupported data format", channel->dataFormat);
                return BTA_StatusNotSupported;
            }
        }
    }
    return BTA_StatusOk;
}


static std::vector<cv::Point2f> scaleVector(std::vector<cv::Point2f> pointBuf, float resize) {
    std::vector<cv::Point2f> ret(pointBuf.size());
    for (int i = 0; i < pointBuf.size(); i++) {
        ret[i].x = pointBuf[i].x * resize;
        ret[i].y = pointBuf[i].y * resize;
    }
    return ret;
}

#endif
